// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#include "gfx/bitmap.h"
#include "gfx/blit.h"
#include "gfx/fastmem.h"
#include "misc/error.h"

void (*CopyRectFunc[Bitmap::Classes])(Bitmap& Dst,int DstX,int DstY,Bitmap& Src,const Rect &SrcRect);

void CopyRect_Invalid(Bitmap& Dst,int dx0,int dy,Bitmap &Src,const Rect &SrcRect);
void CopyRect_General(Bitmap& Dst,int dx0,int dy,Bitmap &Src,const Rect &SrcRect);
void CopyRect_PP8(Bitmap& Dst,int dx0,int dy,Bitmap& Src,const Rect &SrcRect);
static class AutoInitCopyRectFunc
  {
  public:
  AutoInitCopyRectFunc()
    {
    for (int i=0;i<Bitmap::Classes;i++) CopyRectFunc[i]=CopyRect_General;
    CopyRectFunc[Bitmap::ClassInvalid]=CopyRect_Invalid;
    #if defined(ACCEL_CopyRect_PP8) || defined(ACCEL_PP8)
    CopyRectFunc[Bitmap::ClassPackedPixel8]=CopyRect_PP8;
    #endif
    };
  } InitCopyRectFunc;

void CopyRect_Invalid(Bitmap&,int,int,Bitmap&,const Rect&)
  {
  ErrorHandler.Abort("CopyRect  invalid bitmap.");
  }
void CopyRect_General(Bitmap& Dst,int dx0,int dy,Bitmap &Src,const Rect &SrcRect)
  {
  #ifdef DEBUG
  if (Dst.ClassID()!=Src.ClassID() || (Dst.Palette().ClassID()==Palette::ClassColorArray && (&Dst.Palette()!=&Src.Palette()))) ErrorHandler.Abort("CopyRect()  Incompatible bitmaps.");
  int RectWidth=SrcRect.Right-SrcRect.Left+1;
  int RectHeight=SrcRect.Bottom-SrcRect.Top+1;
  if (dx0<0          || dy<0          || dx0+RectWidth>Dst.Width() || dy+RectHeight>Dst.Height()
   || SrcRect.Left<0 || SrcRect.Top<0 || SrcRect.Right>=Src.Width() || SrcRect.Bottom>=Src.Height()) ErrorHandler.Abort("CopyRect  rect exceeds bitmap.");
  ErrorHandler.EnterCriticalSection("CopyRect general"); 
  #endif
  int sy=SrcRect.Top;
  while (sy<=SrcRect.Bottom)
    {
    int sx=SrcRect.Left;
    int dx=dx0;
    while (sx<=SrcRect.Right)
      {
      Dst.SetPixel(dx,dy,Src.GetPixel(sx,sy));
      dx++;
      sx++;
      }
    dy++;
    sy++;
    }
  #ifdef DEBUG
  ErrorHandler.LeaveCriticalSection();
  #endif
  }



#if defined(ACCEL_CopyRect_PP8) || defined(ACCEL_PP8)
void CopyRect_PP8(Bitmap& Dst,int dx0,int dy,Bitmap& Src,const Rect &SrcRect)
  {
  int RectWidth=SrcRect.Right-SrcRect.Left+1;
  int RectHeight=SrcRect.Bottom-SrcRect.Top+1;
  int SrcDiff,DstDiff;
  SrcDiff=Src.WidthBytes()-RectWidth;
  DstDiff=Dst.WidthBytes()-RectWidth;
  #ifdef DEBUG
  if (Dst.ClassID()!=Src.ClassID() || (Dst.Palette().ClassID()==Palette::ClassColorArray && (&Dst.Palette()!=&Src.Palette()))) ErrorHandler.Abort("CopyRect()  Incompatible bitmaps.");
  if (dx0<0          || dy<0          || dx0+RectWidth>Dst.Width() || dy+RectHeight>Dst.Height()
   || SrcRect.Left<0 || SrcRect.Top<0 || SrcRect.Right>=Src.Width() || SrcRect.Bottom>=Src.Height()) ErrorHandler.Abort("CopyRect  rect exceeds bitmap.");
  #endif
  unsigned char *SrcPtr,*DstPtr;
  SrcPtr=Src.Bits()+SrcRect.Top*Src.WidthBytes()+SrcRect.Left;
  DstPtr=Dst.Bits()+dy*Dst.WidthBytes()+dx0;
  FastMemBlockCopy(DstPtr,SrcPtr,RectWidth,DstDiff,SrcDiff,RectHeight);
  }
#endif
