// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#include <i86.h>
#include <dos.h>
#include <conio.h>
#include <iostream.h>
#include "device/timer.h"
#include "misc/error.h"
#include "compiler/types.h"


const int ClockIntNr=0x08;
static int TimerDriver::installeddrivers=0;
TimerDriver SystemTimer;
static AlarmTimer *RootAlarmTimer;

static void RemoveTimerOnError() { SystemTimer.~TimerDriver(); }

static void interrupt timerhandler();
TimerDriver::TimerDriver()
  {
  #ifdef DEBUG
  cout << "init timer ... ";
  #endif
  if (installeddrivers!=0) ErrorHandler.Abort("TimerDriver  is already installed.");
  // recall the initial DOS timer, to restore later the proper time
  // (the DOS int is not chained!).
  {union REGS regs;
  regs.h.ah=0x00;
  int386(0x1a,&regs,&regs);
  DOS_timer=(regs.w.cx<<16)+regs.w.dx;}
  // init new handler
  _disable();
  oldint=_dos_getvect(ClockIntNr);
  _dos_setvect(ClockIntNr,timerhandler);
  // set clock rate to 18.2
  outp(0x43, 0x36);     // set count
  outp(0x40, 0x00);     //
  outp(0x40, 0x00);     // 0 = 65536 = 18.2 calls per second (minimum)
  time=0;
  timeval=55;
  installeddrivers++;
  ErrorHandler.AddCleanUpProc(RemoveTimerOnError);
  _enable();
  #ifdef DEBUG
  cout << "done\n";
  #endif
  }
TimerDriver::~TimerDriver()
  {
  if (!installeddrivers) return;
  #ifdef DEBUG
  cout << "destructing time driver .. ";
  #endif
  // set original clock int
  _dos_setvect(ClockIntNr,oldint);
  // set the clock rate back to original 18.2 Hz
  _disable();
  outp(0x43, 0x36);
  outp(0x40, 0x00);
  outp(0x40, 0x00);
  _enable();
  installeddrivers--;
  ErrorHandler.RemoveCleanUpProc(RemoveTimerOnError);
  // restore the DOS time, the time is not correct, cause the timer int
  // was not chained.
  {union REGS regs;
  regs.h.ah=0x01;
  DOS_timer+=time/55;                    // add the correct number of 18.2=1000/55 Hz ticks
  regs.w.cx=(WORD)(DOS_timer>>16);
  regs.w.dx=(WORD)DOS_timer;
  int386(0x1a,&regs,&regs);}
  #ifdef DEBUG
  cout << "done\n";
  #endif
  }
int TimerDriver::TimeVal(long MilliSecs)
  {
  if (MilliSecs<=0) MilliSecs=1;
  int Ticks=119318*MilliSecs/100;
  if (Ticks>(1<<16)-1) Ticks=(1<<16)-1;
  _disable();
  timeval=Ticks*100/119318;
  if (Ticks==(1<<16)-1) Ticks=0; // max 18.2 calls per sec = 55 millisecs
  outp(0x43,0x36);
  outp(0x40,Ticks&0xff); // low byte
  outp(0x40,Ticks>>8);   // hi byte
  _enable();
  return(0);
  }



static void interrupt timerhandler()
  {
  SystemTimer.time+=SystemTimer.timeval;
  outp(0x20,0x20); // clear IRQ controller
  // handle AlarmTimers
  AlarmTimer *Timer=RootAlarmTimer;
  while (Timer)
    {
    if (Timer->alarmtime<SystemTimer.time)
      {
      if (!(Timer->flags&AlarmTimer::HandlerIsWorking))
        {
        int MissedCalls=0;
        // set new alarmtime and compute number of missed calls
        if (Timer->alarmtime+Timer->timeval<=SystemTimer.time)  // otherwise normal case, no call has been missed
          {
          // calls have been missed
          MissedCalls=(SystemTimer.time-Timer->alarmtime)/Timer->timeval;
          Timer->alarmtime+=(MissedCalls+1)*Timer->timeval;
          }
        Timer->alarmtime+=Timer->timeval;
        // call handler
        Timer->flags|=AlarmTimer::HandlerIsWorking;
        Timer->TimerHandler(MissedCalls);
        Timer->flags&=~AlarmTimer::HandlerIsWorking;
        }
      }
    Timer=Timer->next;
    }
  }



void AlarmTimer::Start(int TimeIndex)
  {
  if (!IsRunning())
    {
    alarmtime=TimeIndex;
    prev=0;
    next=RootAlarmTimer;
    RootAlarmTimer=this;
    flags|=FlagIsRunning;
    }
  }
void AlarmTimer::Stop()
  {
  if (IsRunning())
    {
    if (prev)
      prev->next=next;               // unlink the timer
    else
      RootAlarmTimer=next;
    if (next) next->prev=prev;
    prev=0;
    flags&=~FlagIsRunning;
    }
  }
int AlarmTimer::IsRunning() const  
  { 
  return(flags&FlagIsRunning); 
  };

