/*
	Audio File Library
	Copyright (C) 1998, Michael Pruett <michael@68k.org>

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Library General Public
	License as published by the Free Software Foundation; either
	version 2 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Library General Public License for more details.

	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the 
	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
	Boston, MA  02111-1307  USA.
*/

/*
	swapblock.c

	This file contains functions which efficiently perform byte
	swapping, shifting, and writing.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <assert.h>
#include <stdio.h>
#include "byteorder.h"
#include "swapblock.h"

#define WRITE_BUFFER_SIZE 65536

/*
	fp is the file pointer to the file being written to.
	data is a pointer to an array of integers.
	count is the number of such integers contained in data.
*/

/*
	_af_writeswapblock16 returns the number of byte-swapped 16-bit integers
	which were written to the file pointer fp.
*/

ssize_t _af_writeswapblock16 (FILE *fp, void *data, size_t count)
{
	size_t		i;
	u_int16_t	sb[WRITE_BUFFER_SIZE], *s;
	ssize_t		accumulatedResult = 0, result;

	s = (u_int16_t *) data;

	for (i=0; i<count; i++)
	{
		sb[i % WRITE_BUFFER_SIZE] = _af_byteswapint16(s[i]);

		if ((i % WRITE_BUFFER_SIZE) == WRITE_BUFFER_SIZE - 1)
		{
			/* Attempt to write the buffer to the file descriptor. */
			result = fwrite(sb, 1, 2 * WRITE_BUFFER_SIZE, fp);

			/* Exit returning status upon error. */
			if (result < 0)
				return result;

			accumulatedResult += result;

			/*
				Fail if the number of bytes written was less than the
				number requested.
			*/
			if (result != 2 * WRITE_BUFFER_SIZE)
				return (accumulatedResult / 2);
		}
	}

	/* Write the remaining bytes to the file. */
	result = fwrite(sb, 1, 2 * (i % WRITE_BUFFER_SIZE), fp);

	if (result < 0)
		return result;

	accumulatedResult += result;

	return (accumulatedResult / 2);
}

/*
	_af_writeswapblock32 returns the number of byte-swapped 32-bit integers
	which were written to the file pointer fp.
*/

ssize_t _af_writeswapblock32 (FILE *fp, void *data, size_t count)
{
	size_t		i;
	u_int32_t	lb[WRITE_BUFFER_SIZE], *s;
	ssize_t		accumulatedResult = 0, result;

	s = (u_int32_t *) data;

	for (i=0; i<count; i++)
	{
		lb[i % WRITE_BUFFER_SIZE] = _af_byteswapint32(s[i]);

		if ((i % WRITE_BUFFER_SIZE) == WRITE_BUFFER_SIZE - 1)
		{
			/* Attempt to write the buffer to the file descriptor. */
			result = fwrite(lb, 1, 4 * WRITE_BUFFER_SIZE, fp);

			/* Exit returning status upon error. */
			if (result < 0)
				return result;

			accumulatedResult += result;

			/*
				Fail if the number of bytes written was less than the
				number requested.
			*/
			if (result != 4 * WRITE_BUFFER_SIZE)
				return (accumulatedResult / 4);
		}
	}

	/* Write the remaining bytes to the file. */
	result = fwrite(lb, 1, 4 * (i % WRITE_BUFFER_SIZE), fp);

	if (result < 0)
		return result;

	accumulatedResult += result;

	return (accumulatedResult / 4);
}

/* _af_swapblock16 performs an in-place byte swap on a block of 16-bit integers. */
void _af_swapblock16 (void *data, size_t count)
{
	size_t		i;
	u_int16_t	*sb = data;

	for (i=0; i<count; i++)
	{
		sb[i] = _af_byteswapint16(sb[i]);
	}
}

/* _af_swapblock32 performs an in-place byte swap on a block of 32-bit integers. */
void _af_swapblock32 (void *data, size_t count)
{
	size_t		i;
	u_int32_t	*lb = data;

	for (i=0; i<count; i++)
	{
		lb[i] = _af_byteswapint32(lb[i]);
	}
}

/*
	_af_writecompress4to3 writes the 24-bit quantities with 4-byte alignment
	contained in 'data' to the file referenced by 'fp' as 3-byte
	samples, discarding the highest-order byte.
*/

ssize_t _af_writecompress4to3 (FILE *fp, void *data, size_t count, int swap)
{
	int			i;
	u_int32_t	*s;

	s = (u_int32_t *) data;

	for (i=0; i<count; i++)
	{
		u_int8_t	x[3];
		x[0] = s[i] >> 16;
		x[1] = s[i] >> 8;
		x[2] = s[i];
		fwrite(x, 1, 3, fp);
	}

	return i; /* FIXME: what is proper result? */
}
