/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of Sound Blaster CD/Pro mixer
 *  TODO: Routines for CTL1335 mixer (Sound Blaster 2.0 CD).
 *        If you have this soundcard, mail me!!!
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#include "../../include/driver.h"
#include "../../include/sb.h"

void snd_sb8mixer_write(sbmixer_t * mixer,
		        unsigned char reg, unsigned char data)
{
	outb(reg, SBP1(mixer->port, MIXER_ADDR));
	snd_delay(1);
	outb(data, SBP1(mixer->port, MIXER_DATA));
	snd_delay(1);
}

unsigned char snd_sb8mixer_read(sbmixer_t * mixer, unsigned char reg)
{
	unsigned char result;

	outb(reg, SBP1(mixer->port, MIXER_ADDR));
	snd_delay(1);
	result = inb(SBP1(mixer->port, MIXER_DATA));
	snd_delay(1);
	return result;
}

static int snd_sb8mixer_input_mux(void *private_data,
				  int w_flag,
				  snd_kmixer_element_t **element)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;
	unsigned long flags;
	unsigned char reg, oval, nval;
	int change = 0;

	spin_lock_irqsave(&sbmix->lock, flags);
	reg = snd_sb8mixer_read(sbmix, SB_DSP_CAPTURE_SOURCE);
	oval = reg & 6;
	if (w_flag) {
		nval = 0;
		if (*element == sbmix->me_mux_mic) {
			nval = SB_DSP_MIXS_MIC;
		} else if (*element == sbmix->me_mux_line) {
			nval = SB_DSP_MIXS_LINE;
		} else if (*element == sbmix->me_mux_cd) {
			nval = SB_DSP_MIXS_CD;
		} else {
			spin_unlock_irqrestore(&sbmix->lock, flags);
			return -EINVAL;
		}
		change = nval != oval;
		snd_sb8mixer_write(sbmix, SB_DSP_CAPTURE_SOURCE, (oval & ~6) | nval);
	} else {
		switch (oval) {
		case SB_DSP_MIXS_LINE:
			*element = sbmix->me_mux_line;
			break;
		case SB_DSP_MIXS_CD:
			*element = sbmix->me_mux_cd;
			break;
		case SB_DSP_MIXS_MIC:
		default:
			*element = sbmix->me_mux_mic;
			break;
		}
	}	
	spin_unlock_irqrestore(&sbmix->lock, flags);
	return change;
}

static int snd_sb8mixer_stereo_volume_level(sbmixer_t *sbmix,
					    int w_flag,
					    int *voices,
					    unsigned char reg,
					    unsigned char max,
					    unsigned char left_shift,
					    unsigned char right_shift)
{
	unsigned long flags;
	int change = 0;
	unsigned char oval;
	int left, right;

	spin_lock_irqsave(&sbmix->lock, flags);
	oval = snd_sb8mixer_read(sbmix, reg);
	left = (oval >> left_shift) & max;
	right = (oval >> right_shift) & max;
	if (!w_flag) {
		voices[0] = left;
		voices[1] = right;
	} else {
		change = voices[0] != left && voices[1] != left;
		oval &= ~(max << left_shift);
		oval &= ~(max << right_shift);
		oval |= voices[0] << left_shift;
		oval |= voices[1] << right_shift;
		snd_sb8mixer_write(sbmix, reg, oval);
	}
	spin_unlock_irqrestore(&sbmix->lock, flags);
	return change;
}

static int snd_sb8mixer_mono_volume_level(sbmixer_t *sbmix,
					  int w_flag,
					  int *voices,
					  unsigned char reg,
					  unsigned char max,
					  unsigned char shift)
{
	unsigned long flags;
	int change = 0;
	unsigned char oval;
	int val;

	spin_lock_irqsave(&sbmix->lock, flags);
	oval = snd_sb8mixer_read(sbmix, reg);
	val = (oval >> shift) & max;
	if (!w_flag) {
		voices[0] = val;
	} else {
		change = voices[0] != val;
		oval &= ~(max << shift);
		oval |= voices[0] << shift;
		snd_sb8mixer_write(sbmix, reg, oval);
	}
	spin_unlock_irqrestore(&sbmix->lock, flags);
	return change;
}

static int snd_sb8mixer_master_volume(void *private_data, int w_flag, int *voices)
{
	return snd_sb8mixer_stereo_volume_level((sbmixer_t *)private_data,
						w_flag, voices,
						SB_DSP_MASTER_DEV, 7, 5, 1);
}

static int snd_sb8mixer_pcm_volume(void *private_data, int w_flag, int *voices)
{
	return snd_sb8mixer_stereo_volume_level((sbmixer_t *)private_data,
						w_flag, voices,
						SB_DSP_PCM_DEV, 7, 5, 1);
}

static int snd_sb8mixer_line_volume(void *private_data, int w_flag, int *voices)
{
	return snd_sb8mixer_stereo_volume_level((sbmixer_t *)private_data,
						w_flag, voices,
						SB_DSP_LINE_DEV, 7, 5, 1);
}

static int snd_sb8mixer_cd_volume(void *private_data, int w_flag, int *voices)
{
	return snd_sb8mixer_stereo_volume_level((sbmixer_t *)private_data,
						w_flag, voices,
						SB_DSP_CD_DEV, 7, 5, 1);
}

static int snd_sb8mixer_fm_volume(void *private_data, int w_flag, int *voices)
{
	return snd_sb8mixer_stereo_volume_level((sbmixer_t *)private_data,
						w_flag, voices,
						SB_DSP_FM_DEV, 7, 5, 1);
}

static int snd_sb8mixer_mic_volume(void *private_data, int w_flag, int *voices)
{
	return snd_sb8mixer_mono_volume_level((sbmixer_t *)private_data,
					      w_flag, voices,
					      SB_DSP_MIC_DEV, 3, 1);
}

static int snd_sb8mixer_group_ctrl1(sbmixer_t *sbmix,
				    snd_kmixer_file_t * file,
				    int w_flag,
				    snd_mixer_group_t * ugroup,
				    snd_mixer_volume1_control_t *volume1,
				    snd_kmixer_element_t *volume1_element,
				    int max,
				    snd_kmixer_element_t *mux_in)
{
	int voices[2];
	snd_kmixer_element_t *element;
	int change = 0;

	if (!w_flag) {
		ugroup->channels = SND_MIXER_CHN_MASK_STEREO;
		ugroup->caps = SND_MIXER_GRPCAP_VOLUME;
		volume1(sbmix, 0, voices);
		ugroup->volume.names.front_left = voices[0];
		ugroup->volume.names.front_right = voices[1];
		ugroup->min = 0;
		ugroup->max = max;
		if (mux_in) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE |
			                SND_MIXER_GRPCAP_JOINTLY_CAPTURE |
			                SND_MIXER_GRPCAP_EXCL_CAPTURE;
			ugroup->capture_group = 1;
			snd_sb8mixer_input_mux(sbmix, 0, &element);
			ugroup->capture = 0;
			if (element == mux_in)
				ugroup->capture |= SND_MIXER_CHN_MASK_STEREO;
		}
	} else {
		voices[0] = ugroup->volume.names.front_left & max;
		voices[1] = ugroup->volume.names.front_right & max;
		if (volume1(sbmix, 1, voices) > 0) {
			snd_mixer_element_value_change(file, volume1_element, 0);
			change = 1;
		}
		if (mux_in) {
			snd_sb8mixer_input_mux(sbmix, 0, &element);
			if (ugroup->capture & SND_MIXER_CHN_MASK_STEREO)
				element = mux_in;
			if (snd_sb8mixer_input_mux(sbmix, 1, &element) > 0) {
				snd_mixer_element_value_change(file, sbmix->me_mux, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_sb8mixer_group_ctrl2(sbmixer_t *sbmix,
				    snd_kmixer_file_t * file,
				    int w_flag,
				    snd_mixer_group_t * ugroup,
				    snd_mixer_volume1_control_t *volume1,
				    snd_kmixer_element_t *volume1_element,
				    int max,
				    snd_kmixer_element_t *mux_in)
{
	int voice;
	snd_kmixer_element_t *element;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = SND_MIXER_GRPCAP_VOLUME;
		ugroup->channels = SND_MIXER_CHN_MASK_MONO;
		volume1(sbmix, 0, &voice);
		ugroup->volume.names.front_left = voice;
		ugroup->min = 0;
		ugroup->max = max;
		if (mux_in) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE | SND_MIXER_GRPCAP_EXCL_CAPTURE;
			ugroup->capture_group = 1;
			snd_sb8mixer_input_mux(sbmix, 0, &element);
			ugroup->capture = 0;
			if (element == mux_in)
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_LEFT;
		}
	} else {
		voice = ugroup->volume.names.front_left & max;
		if (volume1(sbmix, 1, &voice) > 0) {
			snd_mixer_element_value_change(file, volume1_element, 0);
			change = 1;
		}
		if (mux_in) {
			snd_sb8mixer_input_mux(sbmix, 0, &element);
			if (ugroup->capture & SND_MIXER_CHN_MASK_STEREO)
				element = mux_in;
			if (snd_sb8mixer_input_mux(sbmix, 1, &element) > 0) {
				snd_mixer_element_value_change(file, sbmix->me_mux, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_sb8mixer_group_master(void *private_data,
				     snd_kmixer_file_t * file,
				     int w_flag,
				     snd_mixer_group_t * ugroup)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;

	return snd_sb8mixer_group_ctrl1(sbmix, file, w_flag, ugroup,
					snd_sb8mixer_master_volume,
					sbmix->me_vol_master,
					7,
					NULL);
}

static int snd_sb8mixer_group_line(void *private_data,
				   snd_kmixer_file_t * file,
				   int w_flag,
				   snd_mixer_group_t * ugroup)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;

	return snd_sb8mixer_group_ctrl1(sbmix, file, w_flag, ugroup,
					snd_sb8mixer_line_volume,
					sbmix->me_vol_line,
					7,
					sbmix->me_mux_line);
}

static int snd_sb8mixer_group_pcm(void *private_data,
				  snd_kmixer_file_t * file,
				  int w_flag,
				  snd_mixer_group_t * ugroup)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;

	return snd_sb8mixer_group_ctrl1(sbmix, file, w_flag, ugroup,
					snd_sb8mixer_pcm_volume,
					sbmix->me_vol_pcm,
					7,
					NULL);
}

static int snd_sb8mixer_group_cd(void *private_data,
				 snd_kmixer_file_t * file,
				 int w_flag,
				 snd_mixer_group_t * ugroup)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;

	return snd_sb8mixer_group_ctrl1(sbmix, file, w_flag, ugroup,
					snd_sb8mixer_cd_volume,
					sbmix->me_vol_cd,
					7,
					sbmix->me_mux_cd);
}

static int snd_sb8mixer_group_fm(void *private_data,
				 snd_kmixer_file_t * file,
				 int w_flag,
				 snd_mixer_group_t * ugroup)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;

	return snd_sb8mixer_group_ctrl1(sbmix, file, w_flag, ugroup,
					snd_sb8mixer_fm_volume,
					sbmix->me_vol_synth,
					7,
					NULL);
}

static int snd_sb8mixer_group_mic(void *private_data,
				  snd_kmixer_file_t * file,
				  int w_flag,
				  snd_mixer_group_t * ugroup)
{
	sbmixer_t *sbmix = (sbmixer_t *)private_data;

	return snd_sb8mixer_group_ctrl2(sbmix, file, w_flag, ugroup,
					snd_sb8mixer_mic_volume,
					sbmix->me_vol_mic,
					3,
					sbmix->me_mux_mic);
}

snd_kmixer_t *snd_sb8dsp_new_mixer(snd_card_t * card,
				   sbmixer_t * sbmix,
				   unsigned short hardware,
				   int pcm_dev)
{
	snd_kmixer_t *mixer;
	unsigned long flags;
	snd_kmixer_group_t *group;
	snd_kmixer_element_t *element;
	static struct snd_mixer_element_volume1_range db_range1[2] = {
		{0, 7, -4600, 0},
		{0, 7, -4600, 0}
	};
	static struct snd_mixer_element_volume1_range db_range2[1] = {
		{0, 3, -4600, 0},
	};

	if (!card)
		return NULL;
	mixer = snd_mixer_new(card, "CTL1345");
	if (!mixer)
		return NULL;
	strcpy(mixer->name, mixer->id);

	spin_lock_irqsave(&sbmix->lock, flags);
	snd_sb8mixer_write(sbmix, 0x00, 0x00);	/* mixer reset */
	spin_unlock_irqrestore(&sbmix->lock, flags);

	/* build input mux and accumulator */
	if ((sbmix->me_mux = snd_mixer_lib_mux2(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, 0, snd_sb8mixer_input_mux, sbmix)) == NULL)
		goto __error;
	if ((sbmix->me_out_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	/* buid capture endpoint */
	if ((element = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_ADC, 0, SND_MIXER_ETYPE_ADC, 8)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_mux, element) < 0)
		goto __error;
 	if ((sbmix->me_capture = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE, 1, &pcm_dev)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, sbmix->me_capture) < 0)
		goto __error;
	/* build master volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_OSS_VOLUME, snd_sb8mixer_group_master, sbmix)) == NULL)
		goto __error;
 	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if ((sbmix->me_vol_master = snd_mixer_lib_volume1(mixer, "Master Volume", 0, 2, db_range1, snd_sb8mixer_master_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_vol_master) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_out_accu, sbmix->me_vol_master) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_master, element) < 0)
		goto __error;
	/* build PCM volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_PCM, 0, SND_MIXER_OSS_PCM, snd_sb8mixer_group_pcm, sbmix)) == NULL)
		goto __error;
 	if ((sbmix->me_playback = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK, 1, &pcm_dev)) == NULL)
		goto __error;
	if ((element = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_DAC, 0, SND_MIXER_ETYPE_DAC, 8)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_playback, element) < 0)
		goto __error;
	if ((sbmix->me_vol_pcm = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, db_range1, snd_sb8mixer_pcm_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_vol_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, sbmix->me_vol_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_pcm, sbmix->me_out_accu) < 0)
		goto __error;
	/* build LINE volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_OSS_LINE, snd_sb8mixer_group_line, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_mux) < 0)
		goto __error;
 	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((sbmix->me_vol_line = snd_mixer_lib_volume1(mixer, "Line Volume", 0, 2, db_range1, snd_sb8mixer_line_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_vol_line) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_line, sbmix->me_out_accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_line, sbmix->me_mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, sbmix->me_vol_line) < 0)
		goto __error;
	sbmix->me_mux_line = sbmix->me_vol_line;
	/* build CD volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_CD, 0, SND_MIXER_OSS_CD, snd_sb8mixer_group_cd, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_mux) < 0)
		goto __error;
 	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_CD, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((sbmix->me_vol_cd = snd_mixer_lib_volume1(mixer, "CD Volume", 0, 2, db_range1, snd_sb8mixer_cd_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_vol_cd) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_cd, sbmix->me_out_accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_cd, sbmix->me_mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, sbmix->me_vol_cd) < 0)
		goto __error;
	sbmix->me_mux_cd = sbmix->me_vol_cd;
	/* build FM volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_FM, 0, SND_MIXER_OSS_SYNTH, snd_sb8mixer_group_fm, sbmix)) == NULL)
		goto __error;
 	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_FM, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((sbmix->me_vol_synth = snd_mixer_lib_volume1(mixer, "FM Volume", 0, 2, db_range1, snd_sb8mixer_fm_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_vol_synth) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_synth, sbmix->me_out_accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, sbmix->me_vol_synth) < 0)
		goto __error;
	/* build MIC volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_OSS_MIC, snd_sb8mixer_group_mic, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_mux) < 0)
		goto __error;
 	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((sbmix->me_vol_mic = snd_mixer_lib_volume1(mixer, "MIC Volume", 0, 1, db_range2, snd_sb8mixer_mic_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, sbmix->me_vol_mic) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_mic, sbmix->me_out_accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, sbmix->me_vol_mic, sbmix->me_mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, sbmix->me_vol_mic) < 0)
		goto __error;
	sbmix->me_mux_mic = sbmix->me_vol_mic;
	
	return mixer;

      __error:
      	snd_mixer_free(mixer);
      	return NULL;
}
